/*
 * Decompiled with CFR 0.152.
 */
package com.google.cloud.bigtable.gaxx.grpc;

import com.google.api.core.ApiFuture;
import com.google.auto.value.AutoValue;
import com.google.bigtable.v2.PingAndWarmResponse;
import com.google.cloud.bigtable.data.v2.stub.BigtableChannelPrimer;
import com.google.cloud.bigtable.gaxx.grpc.AutoValue_ChannelPoolHealthChecker_ProbeResult;
import com.google.cloud.bigtable.gaxx.grpc.BigtableChannelPool;
import com.google.cloud.bigtable.gaxx.grpc.ChannelPrimer;
import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.collect.ImmutableList;
import com.google.common.util.concurrent.MoreExecutors;
import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.util.Comparator;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.Nullable;

class ChannelPoolHealthChecker {
    private static final Logger logger = Logger.getLogger(ChannelPoolHealthChecker.class.getName());
    private static final Duration WINDOW_DURATION = Duration.ofMinutes(5L);
    private static final Duration PROBE_INTERVAL = Duration.ofSeconds(30L);
    @VisibleForTesting
    static final Duration PROBE_DEADLINE = Duration.ofMillis(500L);
    private static final Duration MIN_EVICTION_INTERVAL = Duration.ofMinutes(10L);
    private static final int MIN_PROBES_FOR_EVALUATION = 4;
    private static final int SINGLE_CHANNEL_FAILURE_PERCENT_THRESHOLD = 60;
    private static final int POOLWIDE_BAD_CHANNEL_CIRCUITBREAKER_PERCENT = 70;
    private final Supplier<ImmutableList<BigtableChannelPool.Entry>> entrySupplier;
    private volatile Instant lastEviction;
    private final ScheduledExecutorService executor;
    private final ChannelPrimer channelPrimer;
    private ScheduledFuture<?> probeTaskScheduledFuture;
    private ScheduledFuture<?> detectAndRemoveTaskScheduledFuture;
    private final Clock clock;

    public ChannelPoolHealthChecker(Supplier<ImmutableList<BigtableChannelPool.Entry>> entrySupplier, ChannelPrimer channelPrimer, ScheduledExecutorService executor, Clock clock) {
        this.entrySupplier = entrySupplier;
        this.lastEviction = Instant.MIN;
        this.channelPrimer = channelPrimer;
        this.executor = executor;
        this.clock = clock;
    }

    void start() {
        if (!(this.channelPrimer instanceof BigtableChannelPrimer)) {
            logger.log(Level.WARNING, "Provided channelPrimer not an instance of BigtableChannelPrimer, not checking channel health.");
            return;
        }
        Duration initialDelayProbe = Duration.ofMillis(ThreadLocalRandom.current().nextLong(PROBE_INTERVAL.toMillis()));
        this.probeTaskScheduledFuture = this.executor.scheduleAtFixedRate(this::runProbes, initialDelayProbe.toMillis(), PROBE_INTERVAL.toMillis(), TimeUnit.MILLISECONDS);
        Duration initialDelayDetect = Duration.ofMillis(ThreadLocalRandom.current().nextLong(PROBE_INTERVAL.toMillis()));
        this.detectAndRemoveTaskScheduledFuture = this.executor.scheduleAtFixedRate(this::detectAndRemoveOutlierEntries, initialDelayDetect.toMillis(), PROBE_INTERVAL.toMillis(), TimeUnit.MILLISECONDS);
    }

    public void stop() {
        if (this.probeTaskScheduledFuture != null) {
            this.probeTaskScheduledFuture.cancel(false);
        }
        if (this.detectAndRemoveTaskScheduledFuture != null) {
            this.detectAndRemoveTaskScheduledFuture.cancel(false);
        }
    }

    @VisibleForTesting
    void runProbes() {
        Preconditions.checkState((boolean)(this.channelPrimer instanceof BigtableChannelPrimer), (String)"Health checking can only be enabled with BigtableChannelPrimer, found %s", (Object)this.channelPrimer);
        BigtableChannelPrimer primer = (BigtableChannelPrimer)this.channelPrimer;
        for (BigtableChannelPool.Entry entry : this.entrySupplier.get()) {
            ApiFuture<PingAndWarmResponse> probeFuture = primer.sendPrimeRequestsAsync(entry.getManagedChannel());
            probeFuture.addListener(() -> this.onComplete(entry, this.clock.instant(), probeFuture), MoreExecutors.directExecutor());
        }
    }

    @VisibleForTesting
    void onComplete(BigtableChannelPool.Entry entry, Instant startTime, ApiFuture<PingAndWarmResponse> probeFuture) {
        boolean success;
        try {
            probeFuture.get(PROBE_DEADLINE.toMillis(), TimeUnit.MILLISECONDS);
            success = true;
        }
        catch (Exception e) {
            success = false;
            logger.log(Level.WARNING, "Probe failed", e);
        }
        this.addProbeResult(entry, ProbeResult.create(startTime, success));
    }

    @VisibleForTesting
    void addProbeResult(BigtableChannelPool.Entry entry, ProbeResult result) {
        entry.probeHistory.add(result);
        if (result.isSuccessful()) {
            entry.successfulProbesInWindow.incrementAndGet();
        } else {
            entry.failedProbesInWindow.incrementAndGet();
        }
        this.pruneHistory(entry);
    }

    @VisibleForTesting
    void pruneHistory(BigtableChannelPool.Entry entry) {
        Instant windowStart = this.clock.instant().minus(WINDOW_DURATION);
        while (!entry.probeHistory.isEmpty() && entry.probeHistory.peek().startTime().isBefore(windowStart)) {
            ProbeResult removedResult = entry.probeHistory.poll();
            if (removedResult.isSuccessful()) {
                entry.successfulProbesInWindow.decrementAndGet();
                continue;
            }
            entry.failedProbesInWindow.decrementAndGet();
        }
    }

    @VisibleForTesting
    boolean isEntryHealthy(BigtableChannelPool.Entry entry) {
        int failedProbes = entry.failedProbesInWindow.get();
        int totalProbes = failedProbes + entry.successfulProbesInWindow.get();
        if (totalProbes < 4) {
            return true;
        }
        double failureRate = (double)failedProbes / (double)totalProbes * 100.0;
        return failureRate < 60.0;
    }

    @Nullable
    @VisibleForTesting
    BigtableChannelPool.Entry findOutlierEntry() {
        List unhealthyEntries = this.entrySupplier.get().stream().filter(entry -> !this.isEntryHealthy((BigtableChannelPool.Entry)entry)).collect(Collectors.toList());
        int poolSize = this.entrySupplier.get().size();
        if (unhealthyEntries.isEmpty() || poolSize == 0) {
            return null;
        }
        double unhealthyPercent = (double)unhealthyEntries.size() / (double)poolSize * 100.0;
        if (unhealthyPercent >= 70.0) {
            return null;
        }
        return unhealthyEntries.stream().max(Comparator.comparingInt(entry -> entry.failedProbesInWindow.get())).orElse(null);
    }

    @VisibleForTesting
    void detectAndRemoveOutlierEntries() {
        if (this.clock.instant().isBefore(this.lastEviction.plus(MIN_EVICTION_INTERVAL))) {
            return;
        }
        BigtableChannelPool.Entry outlier = this.findOutlierEntry();
        if (outlier != null) {
            this.lastEviction = this.clock.instant();
            outlier.failedProbesInWindow.set(0);
            outlier.successfulProbesInWindow.set(0);
            outlier.probeHistory.clear();
            outlier.getManagedChannel().enterIdle();
        }
    }

    @AutoValue
    static abstract class ProbeResult {
        ProbeResult() {
        }

        abstract Instant startTime();

        abstract boolean isSuccessful();

        static ProbeResult create(Instant startTime, boolean success) {
            return new AutoValue_ChannelPoolHealthChecker_ProbeResult(startTime, success);
        }
    }
}

